# data_manager.py

import os
import json
import logging
from datetime import datetime, timedelta
import tiktoken  # برای شمارش توکن‌ها

# --- تنظیمات مسیر فایل‌ها ---
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
DATA_FILE = os.path.join(BASE_DIR, "bot_data.json")
LOG_FILE = os.path.join(BASE_DIR, "bot.log")

# --- کش داده‌های گلوبال ---
DATA = {
    "users": {},
    "banned_users": set(),
    "stats": {
        "total_messages": 0,
        "total_users": 0,
        "avg_response_time": 0.0,
        "max_response_time": 0.0,
        "min_response_time": float('inf'),
        "total_responses": 0
    },
    "welcome_message": "سلام {user_mention}! 🤖\n\nمن یک ربات هوشمند هستم. هر سوالی دارید بپرسید.",
    "goodbye_message": "کاربر {user_mention} گروه را ترک کرد. خداحافظ!",
    "maintenance_mode": False,
    "blocked_words": [],
    "scheduled_broadcasts": [],
    "bot_start_time": datetime.now().strftime('%Y-%m-%d %H:%M:%S')
}

logger = logging.getLogger(__name__)

# تابع برای شمارش توکن‌ها
def count_tokens(text: str) -> int:
    """شمارش تعداد توکن‌های متن با استفاده از tokenizer"""
    try:
        # استفاده از tokenizer برای gpt-3.5-turbo
        encoding = tiktoken.get_encoding("cl100k_base")
        return len(encoding.encode(text))
    except Exception as e:
        logger.warning(f"Error counting tokens: {e}, using fallback")
        # تقریب: هر 4 کاراکتر = 1 توکن
        return max(1, len(text) // 4)

def load_data():
    """داده‌ها را از فایل JSON بارگذاری کرده و در کش گلوبال ذخیره می‌کند."""
    global DATA
    try:
        if not os.path.exists(DATA_FILE):
            logger.info(f"فایل داده در {DATA_FILE} یافت نشد. یک فایل جدید ایجاد می‌شود.")
            save_data()
            return

        with open(DATA_FILE, 'r', encoding='utf-8') as f:
            loaded_data = json.load(f)
            loaded_data['banned_users'] = set(loaded_data.get('banned_users', []))
            
            # اطمینان از وجود کلیدهای جدید در فایل‌های قدیمی
            if 'blocked_words' not in loaded_data: loaded_data['blocked_words'] = []
            if 'scheduled_broadcasts' not in loaded_data: loaded_data['scheduled_broadcasts'] = []
            if 'maintenance_mode' not in loaded_data: loaded_data['maintenance_mode'] = False
            if 'bot_start_time' not in loaded_data: loaded_data['bot_start_time'] = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
            if 'avg_response_time' not in loaded_data['stats']:
                loaded_data['stats']['avg_response_time'] = 0.0
                loaded_data['stats']['max_response_time'] = 0.0
                loaded_data['stats']['min_response_time'] = float('inf')
                loaded_data['stats']['total_responses'] = 0
            
            # اطمینان از وجود context برای کاربران قدیمی
            for user_id in loaded_data.get('users', {}):
                if 'context' not in loaded_data['users'][user_id]:
                    loaded_data['users'][user_id]['context'] = []

            DATA.update(loaded_data)
            logger.info(f"داده‌ها با موفقیت از {DATA_FILE} بارگذاری شدند.")

    except json.JSONDecodeError as e:
        logger.error(f"خطا در خواندن JSON از {DATA_FILE}: {e}. ربات با داده‌های اولیه شروع به کار می‌کند.")
    except Exception as e:
        logger.error(f"خطای غیرمنتظره هنگام بارگذاری داده‌ها: {e}. ربات با داده‌های اولیه شروع به کار می‌کند.")

def save_data():
    """کش گلوبال داده‌ها را در فایل JSON ذخیره می‌کند."""
    global DATA
    try:
        data_to_save = DATA.copy()
        data_to_save['banned_users'] = list(DATA['banned_users'])
        
        with open(DATA_FILE, 'w', encoding='utf-8') as f:
            json.dump(data_to_save, f, indent=4, ensure_ascii=False)
        logger.debug(f"داده‌ها با موفقیت در {DATA_FILE} ذخیره شدند.")
    except Exception as e:
        logger.error(f"خطای مهلک: امکان ذخیره داده‌ها در {DATA_FILE} وجود ندارد. خطا: {e}")

def update_user_stats(user_id: int, user):
    """آمار کاربر را پس از هر پیام به‌روز کرده و داده‌ها را ذخیره می‌کند."""
    global DATA
    now_str = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    user_id_str = str(user_id)
    
    if user_id_str not in DATA['users']:
        DATA['users'][user_id_str] = {
            'first_name': user.first_name,
            'username': user.username,
            'first_seen': now_str,
            'message_count': 0,
            'context': []  # اضافه کردن context خالی برای کاربر جدید
        }
        DATA['stats']['total_users'] += 1
        logger.info(f"کاربر جدید ثبت شد: {user_id} ({user.first_name})")

    DATA['users'][user_id_str]['last_seen'] = now_str
    DATA['users'][user_id_str]['message_count'] += 1
    DATA['stats']['total_messages'] += 1
    
    save_data()

def update_response_stats(response_time: float):
    """آمار زمان پاسخگویی را به‌روز می‌کند."""
    global DATA
    
    if DATA['stats']['total_responses'] == 0:
        DATA['stats']['min_response_time'] = response_time

    DATA['stats']['total_responses'] += 1
    
    # محاسبه میانگین جدید
    current_avg = DATA['stats']['avg_response_time']
    total_responses = DATA['stats']['total_responses']
    new_avg = ((current_avg * (total_responses - 1)) + response_time) / total_responses
    DATA['stats']['avg_response_time'] = new_avg
    
    # به‌روزرسانی حداکثر و حداقل زمان پاسخگویی
    if response_time > DATA['stats']['max_response_time']:
        DATA['stats']['max_response_time'] = response_time
    
    if response_time < DATA['stats']['min_response_time']:
        DATA['stats']['min_response_time'] = response_time
    
    save_data()

def is_user_banned(user_id: int) -> bool:
    """بررسی می‌کند آیا کاربر مسدود شده است یا خیر."""
    return user_id in DATA['banned_users']

def ban_user(user_id: int):
    """کاربر را مسدود کرده و ذخیره می‌کند."""
    DATA['banned_users'].add(user_id)
    save_data()

def unban_user(user_id: int):
    """مسدودیت کاربر را برداشته و ذخیره می‌کند."""
    DATA['banned_users'].discard(user_id)
    save_data()

def contains_blocked_words(text: str) -> bool:
    """بررسی می‌کند آیا متن حاوی کلمات مسدود شده است یا خیر."""
    if not DATA['blocked_words']:
        return False
    
    text_lower = text.lower()
    for word in DATA['blocked_words']:
        if word in text_lower:
            return True
    
    return False

def get_active_users(days: int) -> list:
    """لیست کاربران فعال در بازه زمانی مشخص را برمی‌گرداند."""
    now = datetime.now()
    cutoff_date = now - timedelta(days=days)
    
    active_users = []
    for user_id, user_info in DATA['users'].items():
        if 'last_seen' in user_info:
            try:
                last_seen = datetime.strptime(user_info['last_seen'], '%Y-%m-%d %H:%M:%S')
                if last_seen >= cutoff_date:
                    active_users.append(int(user_id))
            except ValueError:
                continue
    
    return active_users

def get_users_by_message_count(min_count: int) -> list:
    """لیست کاربران با تعداد پیام بیشتر یا مساوی مقدار مشخص را برمی‌گرداند."""
    users = []
    for user_id, user_info in DATA['users'].items():
        if user_info.get('message_count', 0) >= min_count:
            users.append(int(user_id))
    
    return users

# --- توابع مدیریت context ---

def add_to_user_context(user_id: int, role: str, content: str):
    """اضافه کردن پیام به context کاربر و محدود کردن به 512 توکن"""
    user_id_str = str(user_id)
    
    if user_id_str not in DATA['users']:
        return
    
    if 'context' not in DATA['users'][user_id_str]:
        DATA['users'][user_id_str]['context'] = []
    
    # اضافه کردن پیام جدید
    message = {
        'role': role,  # 'user' یا 'assistant'
        'content': content,
        'timestamp': datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    }
    
    DATA['users'][user_id_str]['context'].append(message)
    
    # محدود کردن context به 512 توکن
    trim_user_context(user_id)
    
    save_data()

def trim_user_context(user_id: int, max_tokens: int = 512):
    """کوتاه کردن context کاربر به تعداد توکن مشخص"""
    user_id_str = str(user_id)
    
    if user_id_str not in DATA['users'] or 'context' not in DATA['users'][user_id_str]:
        return
    
    context = DATA['users'][user_id_str]['context']
    
    if not context:
        return
    
    # محاسبه توکن‌های کل
    total_tokens = sum(count_tokens(msg['content']) for msg in context)
    
    # حذف قدیمی‌ترین پیام‌ها تا زمانی که توکن‌ها زیر حد مجاز باشد
    while total_tokens > max_tokens and len(context) > 1:
        removed_message = context.pop(0)  # حذف قدیمی‌ترین پیام
        total_tokens -= count_tokens(removed_message['content'])
    
    # اگر هنوز بیشتر از حد مجاز است، از محتوای پیام‌ها کم کن
    if total_tokens > max_tokens and context:
        # از آخرین پیام (جدیدترین) شروع به کوتاه کردن کن
        last_message = context[-1]
        content = last_message['content']
        
        # کوتاه کردن محتوا
        tokens = count_tokens(content)
        if tokens > max_tokens:
            # اگر یک پیام به تنهایی بیشتر از حد مجاز باشد، آن را حذف کن
            context.pop()
        else:
            # در غیر این صورت، محتوای پیام را کوتاه کن
            while tokens > (max_tokens - (total_tokens - tokens)) and content:
                # حذف کلمات از انتها
                words = content.split()
                if len(words) > 1:
                    content = ' '.join(words[:-1])
                else:
                    content = content[:-1] if len(content) > 1 else ''
                tokens = count_tokens(content)
            
            if content:  # اگر محتوایی باقی مانده
                context[-1]['content'] = content
            else:  # اگر تمام محتوا حذف شد
                context.pop()
    
    save_data()

def get_user_context(user_id: int) -> list:
    """دریافت context کاربر"""
    user_id_str = str(user_id)
    
    if user_id_str not in DATA['users']:
        return []
    
    return DATA['users'][user_id_str].get('context', [])

def clear_user_context(user_id: int):
    """پاک کردن context کاربر"""
    user_id_str = str(user_id)
    
    if user_id_str in DATA['users']:
        if 'context' in DATA['users'][user_id_str]:
            DATA['users'][user_id_str]['context'] = []
            save_data()
            logger.info(f"Context cleared for user {user_id}")

def get_context_summary(user_id: int) -> str:
    """خلاصه‌ای از context کاربر"""
    context = get_user_context(user_id)
    if not context:
        return "بدون تاریخچه"
    
    total_messages = len(context)
    total_tokens = sum(count_tokens(msg['content']) for msg in context)
    
    # نمایش آخرین پیام
    last_message = context[-1] if context else {}
    last_content = last_message.get('content', '')[:50]
    if len(last_message.get('content', '')) > 50:
        last_content += "..."
    
    return f"{total_messages} پیام ({total_tokens} توکن) - آخرین: {last_message.get('role', '')}: {last_content}"

def get_context_for_api(user_id: int) -> list:
    """فرمت context برای ارسال به API"""
    context = get_user_context(user_id)
    
    # تبدیل به فرمت مورد نیاز API
    api_context = []
    for msg in context:
        api_context.append({
            'role': msg['role'],
            'content': msg['content']
        })
    
    return api_context

def get_context_token_count(user_id: int) -> int:
    """تعداد کل توکن‌های context کاربر"""
    context = get_user_context(user_id)
    return sum(count_tokens(msg['content']) for msg in context)

# بارگذاری اولیه داده‌ها در زمان ایمپورت شدن ماژول
load_data()